
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2008
   *  Last modified   :  %modify_time%
   */
  /** @file 
      *  \brief This file includes the RC4 ATP tests
   *
   *  \version CRYS_SEP_ATP_HASH_Acceptance.c#1:csrc:1
   *  \author 
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern CRYSError_t      TST_ATP_Error;
/************* Private function prototype ***********************/
/************************ Public Functions ******************************/
CRYSError_t SEPQA_ATP_CheckErrorReturn(CRYSError_t  TST_ErrorRet,
									   char         TST_FuncName[TST_FUNC_STRING_LEN],
									   char        *FailureFunctionNameOut_ptr);

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RC4_NotIntegrated         
* 
* Inputs:
* Key_ptr -  A pointer to the user's key buffer.
* KeySize - The size of the KEY in bytes.
* EncryptDecryptFlag - flag to execute encrypt/decrypt
* OutputLocation - indication if input buffer == output buffer or not.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for RC4 non integrated API's                         
*  This function Encrypts and Decrypts known vectors using RC4 
*  Vector sizes used (in bytes) - 512
*  Key Size used (in bytes)    - 256
*  This also outputs the results to an output buffer or the same input buffer
* Algorithm:       
* 1. Allocate memory buffers
* 2. Call the CRYS_RC4_Init function to start the encryption/decryption process
* 2. Call the CRYS_RC4_Stream - This function is called N times
*                              (N = number of blocks in data each block is 1byte)
* 3. Call the CRYS_RC4_Free to end the encryption/decryption process 
* 4. Compare the outputted result to the expected one 
* 6. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_RC4_NotIntegrated(
									DxUint8_t 		*KeyPtr,
									DxUint16_t		KeySize,
									DxUint8_t       EncryptDecryptFlag,
									DxUint8_t       OutputLocation,
									DxUint8_t       *DataIn_ptr,
									DxUint32_t      DataInSize,
									DxUint8_t       *ExpectedData_ptr,        
									DxUint32_t      DataExpSize,
									char           *FailureFunctionNameOut_ptr)
{
	/*** TST Variables Definitions ****/
	DxUint32_t				Block_index = 0;
	DxUint32_t				NumOfBlocks = 0;
	CRYS_RC4UserContext_t   TST_ContextID_ptr;
	DxUint8_t				*pDataIn;
	DxUint8_t				*pDataOut;
	CRYSError_t             TST_TestStatus=0;

	/* data expected size = data input size */
	DataExpSize = DataInSize;

	/* 1. Allocate memory buffers*/
	/*****************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	NumOfBlocks = DataInSize/TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS;

	/* 2. calling the RC4 init function */
	/************************************/
	TST_ATP_Error = CRYS_RC4_Init(&TST_ContextID_ptr,
		KeyPtr,
		KeySize);                        

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RC4_Init",
		FailureFunctionNameOut_ptr);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 3. calling the RC4 Block function */
	/*************************************/
	if (OutputLocation == 0)/* input buffer != output buffer */
	{
		for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RC4 block loop*/
		{  
			if (EncryptDecryptFlag == TST_ENCRYPT)
			{ 
				TST_ATP_Error = CRYS_RC4_Stream(&TST_ContextID_ptr,    
					pDataIn+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),        
					TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
					pDataOut+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index)); 

				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
					"CRYS_RC4_Stream - Encrypt",
					FailureFunctionNameOut_ptr);                             
			}
			else
			{                           
				TST_ATP_Error = CRYS_RC4_Stream(&TST_ContextID_ptr,    
					ExpectedData_ptr+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),        
					TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
					pDataOut+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index));        

				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
					"CRYS_RC4_Stream - Decrypt",
					FailureFunctionNameOut_ptr);
			}   

			if (TST_TestStatus != TESTPASS) 
				goto End;

		}/*End RC4 block loop*/
	}
	else /* input buffer = output buffer */
	{
		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{
			if (EncryptDecryptFlag == TST_ENCRYPT)
			{
				/*Copying the input buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),
					pDataIn+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),
					TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS);  
			}
			else
			{
				/*Copying the input buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),
					ExpectedData_ptr+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),
					TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS);  
			}
		}           

		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{   
			TST_ATP_Error = CRYS_RC4_Stream(&TST_ContextID_ptr,              
				pDataOut+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),        
				TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
				pDataOut+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index)); 

			if (EncryptDecryptFlag == TST_ENCRYPT)
				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RC4_Stream - Encrypt",
				FailureFunctionNameOut_ptr);
			else
				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_RC4_Stream - Decrypt",
				FailureFunctionNameOut_ptr);
			if (TST_TestStatus != TESTPASS) 
				goto End;
		}
	} 

	/* 4. calling the Free function */
	/********************************/        
	TST_ATP_Error = CRYS_RC4_Free( &TST_ContextID_ptr );

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_RC4_Free",
		FailureFunctionNameOut_ptr);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 4. Compare the result to expected result */
	/********************************************/
	if (EncryptDecryptFlag == TST_ENCRYPT)  
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
			ExpectedData_ptr,
			DataInSize);
	}
	else
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
			pDataIn,
			DataInSize);   

		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
			pDataIn,
			DataInSize);
	}


	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut_ptr);

End:
	/* 6. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;	
}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RC4_Integrated         
* 
* Inputs:
* Key_ptr -  A pointer to the user's key buffer.
* KeySize - The size of the KEY in bytes.
* EncryptDecryptFlag - flag to execute encrypt/decrypt
* OutputLocation - indication if input buffer == output buffer or not.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for RC4 non integrated API's                         
*  This function Encrypts and Decrypts known vectors using RC4 
*  Vector sizes used (in bytes) - 512
*  Key Size used (in bytes)    - 256
*  This also outputs the results to an output buffer or the same input buffer
* Algorithm:       
* 1. Allocate memory buffers
* 2. Call the CRYS_RC4_to Preform the encryption/decryption process
* 3. Compare the outputted result to the expected one                                                   
* 6. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_RC4_Integrated(	DxUint8_t 		*KeyPtr,
									DxUint16_t		KeySize,
									DxUint8_t       EncryptDecryptFlag,
									DxUint8_t       OutputLocation,
									DxUint8_t       *DataIn_ptr,
									DxUint32_t      DataInSize,
									DxUint8_t       *ExpectedData_ptr,        
									DxUint32_t      DataExpSize,
									char           *FailureFunctionNameOut_ptr)
{
	/*** TST Variables Definitions ****/
	DxUint32_t				Block_index=0;
	DxUint32_t				NumOfBlocks =0;
	DxUint8_t				*pDataIn;
	DxUint8_t				*pDataOut;
	CRYSError_t             TST_TestStatus=0;

	/* data expected size = data input size */
	DataExpSize = DataInSize;

	/* 1. Allocate memory buffers*/
	/*****************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	NumOfBlocks = DataInSize/TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS;

	/* 2. calling the RC4 integrated function */
	/******************************************/ 
	if (OutputLocation == 0)/* input buffer != output buffer */
	{
		if (EncryptDecryptFlag ==0) 
		{
			TST_ATP_Error = CRYS_RC4(KeyPtr,
									 KeySize,
									 pDataIn,
									 DataInSize,
									 pDataOut); 
		}
		else
		{
			TST_ATP_Error = CRYS_RC4(KeyPtr,
									 KeySize,
									 ExpectedData_ptr,
									 DataInSize,
									 pDataOut);
		}
	}
	else /* input buffer = output buffer */
	{
		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{
			if (EncryptDecryptFlag == TST_ENCRYPT)
			{
				/*Copying the input buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),
								  pDataIn+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),
								  TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS);  
			}
			else
			{
				/*Copying the input buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),
								  ExpectedData_ptr+(TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS*Block_index),
					              TST_MAX_SINGLE_RC4_BLOCK_SIZE_IN_BYTS);  
			}
		}  
		

		TST_ATP_Error = CRYS_RC4(KeyPtr,
								 KeySize,
								 pDataOut,
								 DataInSize,
								 pDataOut); 

	} 

	if (EncryptDecryptFlag == TST_ENCRYPT)
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RC4 Encrypt",
													FailureFunctionNameOut_ptr);
	else
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_RC4 Decrypt",
													FailureFunctionNameOut_ptr);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	
	/* 3. Compare the result to expected result */
	/********************************************/                                                             
	if (EncryptDecryptFlag == TST_ENCRYPT)  
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
												   ExpectedData_ptr,
												   DataInSize);
	}
	else
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp( pDataOut,
													pDataIn,
													DataInSize);
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut_ptr);
End:
	/* 4. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;	
}










